<?php
/**
 * Plugin Name: TranslatePress Search Debug
 * Description: Debug tool to analyze searchable strings for specific post IDs in TranslatePress dictionary tables
 * Version: 1.0.0
 * Author: Debug Tool
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class TRP_Search_Debug {

    private $db;

    public function __construct() {
        global $wpdb;
        $this->db = $wpdb;

        add_action( 'plugins_loaded', array( $this, 'clear_search_transients' ), 1 );
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
    }

    /**
     * Clear all TranslatePress search-related transients on plugins_loaded.
     * This ensures fresh data when debugging search functionality.
     */
    public function clear_search_transients() {
        // Clear gettext search transient
        delete_transient( 'trp_gettext_search' );

        // Clear any cached post URL processing transients
        // These follow the pattern: processed_original_string_meta_post_id_for_{hash}
        global $wpdb;
        $wpdb->query(
            "DELETE FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_processed_original_string_meta_post_id_for_%'
            OR option_name LIKE '_transient_timeout_processed_original_string_meta_post_id_for_%'"
        );
    }

    public function add_admin_menu() {
        add_management_page(
            'TranslatePress Search Debug',
            'TRP Search Debug',
            'manage_options',
            'trp-search-debug',
            array( $this, 'render_admin_page' )
        );
    }

    public function render_admin_page() {
        $post_id = isset( $_GET['post_id'] ) ? absint( $_GET['post_id'] ) : 0;
        $results = array();
        $post_title = '';

        if ( $post_id > 0 ) {
            $post = get_post( $post_id );
            $post_title = $post ? $post->post_title . ' (' . $post->post_type . ')' : 'Post not found';
            $results = $this->get_searchable_strings_for_post( $post_id );
        }

        ?>
        <div class="wrap">
            <h1>TranslatePress Search Debug</h1>
            <p>Analyze searchable strings for a specific post ID stored in TranslatePress dictionary tables.</p>

            <form method="get" action="">
                <input type="hidden" name="page" value="trp-search-debug" />
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="post_id">Post ID:</label>
                        </th>
                        <td>
                            <input
                                type="number"
                                id="post_id"
                                name="post_id"
                                value="<?php echo esc_attr( $post_id ); ?>"
                                class="regular-text"
                                min="1"
                                required
                            />
                            <p class="description">Enter the WordPress post ID to analyze</p>
                        </td>
                    </tr>
                </table>
                <p class="submit">
                    <input type="submit" name="submit" id="submit" class="button button-primary" value="Analyze Post" />
                </p>
            </form>

            <?php if ( $post_id > 0 ) : ?>

                <hr style="margin: 30px 0;" />

                <h2>Results for Post ID: <?php echo esc_html( $post_id ); ?></h2>
                <?php if ( $post_title ) : ?>
                    <p><strong>Post:</strong> <?php echo esc_html( $post_title ); ?></p>
                <?php endif; ?>

                <?php if ( empty( $results ) ) : ?>
                    <div class="notice notice-warning inline">
                        <p>No searchable strings found for this post ID. This could mean:</p>
                        <ul style="margin-left: 20px;">
                            <li>The post has not been translated yet</li>
                            <li>The post content is not wrapped with trp-post-container tags</li>
                            <li>TranslatePress is not active or configured</li>
                        </ul>
                    </div>
                <?php else : ?>
                    <?php foreach ( $results as $language_code => $data ) : ?>
                        <div class="trp-language-section" style="margin-bottom: 30px; padding: 20px; background: #fff; border: 1px solid #ccd0d4; border-radius: 4px;">
                            <h3 style="margin-top: 0; padding-bottom: 10px; border-bottom: 2px solid #0073aa;">
                                Language: <?php echo esc_html( $language_code ); ?>
                                <span style="font-weight: normal; font-size: 14px; color: #666;">
                                    (<?php echo esc_html( $data['count'] ); ?> strings)
                                </span>
                            </h3>

                            <table class="wp-list-table widefat fixed striped" style="margin-top: 15px;">
                                <thead>
                                <tr>
                                    <th style="width: 5%;">ID</th>
                                    <th style="width: 35%;">Original String</th>
                                    <th style="width: 35%;">Translated String</th>
                                    <th style="width: 15%;">Status</th>
                                    <th style="width: 10%;">Block Type</th>
                                </tr>
                                </thead>
                                <tbody>
                                <?php foreach ( $data['strings'] as $string ) : ?>
                                    <tr>
                                        <td><?php echo esc_html( $string->original_id ); ?></td>
                                        <td>
                                            <div style="max-height: 100px; overflow-y: auto; word-break: break-word;">
                                                <?php echo esc_html( substr( $string->original, 0, 500 ) ); ?>
                                                <?php if ( strlen( $string->original ) > 500 ) : ?>
                                                    <em>... (<?php echo strlen( $string->original ); ?> chars total)</em>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td>
                                            <div style="max-height: 100px; overflow-y: auto; word-break: break-word;">
                                                <?php if ( ! empty( $string->translated ) ) : ?>
                                                    <?php echo esc_html( substr( $string->translated, 0, 500 ) ); ?>
                                                    <?php if ( strlen( $string->translated ) > 500 ) : ?>
                                                        <em>... (<?php echo strlen( $string->translated ); ?> chars total)</em>
                                                    <?php endif; ?>
                                                <?php else : ?>
                                                    <em style="color: #999;">Not translated</em>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td>
                                            <?php
                                            $status_labels = array(
                                                0 => '<span style="color: #dc3232;">Not Translated</span>',
                                                1 => '<span style="color: #0073aa;">Machine Translated</span>',
                                                2 => '<span style="color: #46b450;">Human Reviewed</span>',
                                                3 => '<span style="color: #826eb4;">Similar Translated</span>',
                                            );
                                            echo isset( $status_labels[ $string->status ] ) ? $status_labels[ $string->status ] : esc_html( $string->status );
                                            ?>
                                        </td>
                                        <td>
                                            <?php
                                            $block_labels = array(
                                                0 => 'Regular',
                                                1 => 'Active Block',
                                                2 => 'Deprecated',
                                            );
                                            echo isset( $block_labels[ $string->block_type ] ) ? esc_html( $block_labels[ $string->block_type ] ) : esc_html( $string->block_type );
                                            ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endforeach; ?>

                    <div class="notice notice-info inline" style="margin-top: 20px;">
                        <p><strong>Database Structure Info:</strong></p>
                        <ul style="margin-left: 20px;">
                            <li><strong>Dictionary Tables:</strong> <?php echo esc_html( $this->db->prefix ); ?>trp_dictionary_[language_code]</li>
                            <li><strong>Original Strings:</strong> <?php echo esc_html( $this->db->prefix ); ?>trp_original_strings</li>
                            <li><strong>Meta Table:</strong> <?php echo esc_html( $this->db->prefix ); ?>trp_original_meta</li>
                            <li><strong>Meta Key:</strong> post_parent_id</li>
                        </ul>
                    </div>
                <?php endif; ?>

            <?php endif; ?>
        </div>

        <style>
            .trp-language-section table td,
            .trp-language-section table th {
                padding: 10px;
            }

            .trp-language-section table td div {
                font-family: 'Courier New', monospace;
                font-size: 12px;
                line-height: 1.5;
            }

            .form-table th {
                padding: 20px 10px 20px 0;
            }

            .notice.inline {
                padding: 12px;
                margin: 20px 0;
            }

            .notice.inline ul {
                list-style-type: disc;
            }
        </style>
        <?php
    }

    private function get_searchable_strings_for_post( $post_id ) {
        $results = array();

        // Get TranslatePress settings
        $trp_settings = get_option( 'trp_settings' );
        if ( ! $trp_settings || empty( $trp_settings['translation-languages'] ) ) {
            return $results;
        }

        $translation_languages = $trp_settings['translation-languages'];
        $default_language = $trp_settings['default-language'];
        $meta_table = $this->db->prefix . 'trp_original_meta';

        // Check if meta table exists
        if ( $this->db->get_var( "SHOW TABLES LIKE '$meta_table'" ) != $meta_table ) {
            return $results;
        }

        // Get all original_ids associated with this post
        // Note: meta_value is stored as VARCHAR, so we use %s instead of %d
        $original_ids = $this->db->get_col( $this->db->prepare(
            "SELECT original_id FROM $meta_table WHERE meta_key = 'post_parent_id' AND meta_value = %s",
            $post_id
        ) );

        if ( empty( $original_ids ) ) {
            return $results;
        }

        // For each language, get the strings
        foreach ( $translation_languages as $language_code ) {
            // Table name format: wp_trp_dictionary_{default_language}_{language_code}
            $dictionary_table = $this->db->prefix . 'trp_dictionary_' . strtolower($default_language) . '_' . strtolower($language_code);

            // Check if table exists
            if ( $this->db->get_var( "SHOW TABLES LIKE '$dictionary_table'" ) != $dictionary_table ) {
                continue;
            }

            // Get all strings for this language that match the original_ids
            // Sanitize the IDs (they're already integers from the database)
            $sanitized_ids = array_map( 'absint', $original_ids );
            $ids_list = implode( ', ', $sanitized_ids );

            $query = "SELECT original_id, original, translated, status, block_type
                FROM $dictionary_table
                WHERE original_id IN ( $ids_list )
                ORDER BY original_id ASC";

            $strings = $this->db->get_results( $query );

            if ( ! empty( $strings ) ) {
                $results[ $language_code ] = array(
                    'count' => count( $strings ),
                    'strings' => $strings,
                );
            }
        }

        return $results;
    }
}

// Initialize the plugin
new TRP_Search_Debug();
