<?php
namespace TranslatePressCleanupDb\Admin;

class CleanupDatabase
{
    public function __construct()
    {
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_script']);
        add_action('wp_ajax_translatepress_cleanup_database', [$this, 'handle_ajax_cleanup']);
    }

    public function enqueue_admin_script()
    {
        wp_enqueue_script('tpcleanup-script', TPCLEANUP_URL . '/assets/tpcleanup-script.js', array('jquery'), null, true);
        wp_localize_script('tpcleanup-script', 'tpcleanupAjax', array('ajax_url' => admin_url('admin-ajax.php')));
    }

    public function handle_ajax_cleanup()
    {
        // Check the nonce
        check_ajax_referer('translatepress-cleanup-db', 'security');

        // Check if the current user is an administrator
        if (!current_user_can('manage_options')) {
            wp_send_json_error('You do not have permission to perform this action.');
            wp_die();
        }

        if(!class_exists('\TRP_Translate_Press')){
            return wp_send_json_error("TranslatePress plugin not active. Nothing was executed.");
        }

        $trp = \TRP_Translate_Press::get_trp_instance();

        $settings_component = $trp->get_component( 'settings' );
        $settings      = $settings_component->get_settings();

        $query_component = $trp->get_component( 'query' );

        $default_language = $settings['default-language'];
        $array_of_table_names = $query_component->get_all_table_names( $default_language, array() );
        $originals_table =  $query_component->get_table_name_for_original_strings();

        $array_of_table_names[] = $originals_table;

        $table_name = $_POST['table_name'] ?? $array_of_table_names[0];

        // Get batch size from the request
        $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : 2000;

        // Get list of line separated texts from request
        $line_separated_texts= isset($_POST['line_separated_texts']) ? strval($_POST['line_separated_texts']) : '';

        // We'll use this option to exclude strings from entering translation moving forward.
        update_option('trpc_cleanup_line_separated_texts', $line_separated_texts, true);

        // Cleanup Batch
        $strings_deleted = $this->cleanup_db($table_name, $batch_size, $line_separated_texts);

        $finished = '';
        if ($strings_deleted == 0)
        {
            $key = array_search($table_name, $array_of_table_names);
            if ($key !== false && isset($array_of_table_names[$key + 1]))
            {
                // $table_name is not the last, assign the next value
                $table_name = $array_of_table_names[$key + 1];
            } else {
                // $table_name is the last value or not found
                $finished = 'yes';
            }
        }
        // Return the progress
        $payload = array(
            'table_name'            => $table_name,
            'strings_deleted'       => $strings_deleted,
            'line_separated_texts'  => $line_separated_texts,
            'finished'              => $finished
        );

        // Send response back
        wp_send_json_success($payload);
    }

    private function cleanup_db($table_name, $batch_size, $line_separated_texts):int{
        $trp = \TRP_Translate_Press::get_trp_instance();
        $query_component = $trp->get_component( 'query' );

        $array_with_texts = preg_split('/\r\n|\r|\n/', $line_separated_texts, -1, PREG_SPLIT_NO_EMPTY);

        // Initialize the query and placeholders
        $where_clauses = [];
        $placeholders = array($table_name);

        // Construct the WHERE clause dynamically for LIKE conditions
        if (!empty($array_with_texts)) {
            foreach ($array_with_texts as $text) {
                $where_clauses[] = "original LIKE %s"; // Add LIKE %s for each text
                $placeholders[] = '%' . $text . '%'; // Wrap the text with % for partial match
            }
        } else {
            // we don't have any strings to clean.
            return 0;
        }

        // Combine the WHERE clauses with OR
        $or_query = implode(' OR ', $where_clauses);

        // Append the LIMIT placeholder to the values array
        $placeholders[] = intval($batch_size);
        // Construct the final query using wpdb->prepare

        // Using the splat operator ... https://www.php.net/manual/en/migration56.new-features.php so we can expand the $placeholder array into individual variables.
        $query = $query_component->db->prepare("
        DELETE FROM %i 
        WHERE $or_query 
        ORDER BY id LIMIT %d;", ...$placeholders);

        return intval($query_component->db->query($query_component->db->remove_placeholder_escape($query)));
    }

}