<?php
namespace TranslatePressCleanupDb\Admin;

class CleanupEncodedEmails
{
    private $trp;
    public function __construct()
    {
        if(class_exists('\TRP_Translate_Press')){
            $this->trp = \TRP_Translate_Press::get_trp_instance();
            add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_script']);
            add_action('wp_ajax_translatepress_cleanup_encoded_emails', [$this, 'handle_ajax_cleanup']);
        } else {
            $this->trp = false;
        }
    }

    public function enqueue_admin_script()
    {
        wp_enqueue_script('tpCleanupEncodedEmails-script', TPCLEANUP_URL . '/assets/tpcleanup-encoded-emails-script.js', array('jquery'), null, true);
        wp_localize_script('tpCleanupEncodedEmails-script', 'tpCleanupEncodedEmailsAjax', array('ajax_url' => admin_url('admin-ajax.php')));
    }

    public function handle_ajax_cleanup()
    {
        // Check the nonce
        check_ajax_referer('translatepress-cleanup-encoded-emails', 'security');

        // find the current table_name we're operating on.
        // It can be passed on from $_POST or the first one.
        $array_of_table_names = $this->table_names();
        $table_name = $_POST['table_name'] ?? $array_of_table_names[0];

        // Get batch size from the request
        $batch_size = intval($_POST['batch_size']) ?? 2000;

        // Get last_processed_id from the request
        $last_processed_id = intval($_POST['last_processed_id']) ?? 0;

        // Get list of line separated emails from request
        $line_separated_encoded_emails= strval($_POST['line_separated_encoded_emails']) ?? '';

        // We'll store this so the user can come back and run the script again in the future.
        update_option('trpc_cleanup_line_separated_encoded_emails', $line_separated_encoded_emails, false);

        // Cleanup Batch
        $cleanup_result = $this->cleanup_db($line_separated_encoded_emails, $table_name, $batch_size, $last_processed_id);

        $finished = '';
        if ($cleanup_result['table_end'])
        {
            // determine if we start on a new table
            $key = array_search($table_name, $array_of_table_names);
            if ($key !== false && isset($array_of_table_names[$key + 1]))
            {
                // $table_name is not the last, assign the next value
                $table_name = $array_of_table_names[$key + 1];
                // reset last_processed_id when we change table name
                $cleanup_result['last_processed_id'] = 0;
            } else {
                // $table_name is the last value or not found
                $finished = 'yes';
            }
        }
        // Return the progress
        $payload = array(
            'table_name'            => $table_name,
            'last_processed_id'     => $cleanup_result['last_processed_id'],
            'strings_deleted'       => $cleanup_result['strings_deleted'],
            'finished'              => $finished
        );

        // Send response back
        wp_send_json_success($payload);
    }

    /**
     * Returns an array with number of deleted rows and bool $table_end if $last_processed_id >= MAX(id) for that table, meaning we reached the end of the table, false otherwise
     *
     * @param $line_separated_encoded_emails
     * @param $table_name
     * @param $batch_size
     * @param $last_processed_id
     * @return array ('strings_deleted' => 0, 'last_processed_id' => 123, 'table_end' => false)
     */
    private function cleanup_db($line_separated_encoded_emails, $table_name, $batch_size, $last_processed_id):array{
        $query_component = $this->trp->get_component( 'query' );

        // Init variables.
        $array_with_encoded_emails = preg_split('/\r\n|\r|\n/', $line_separated_encoded_emails, -1, PREG_SPLIT_NO_EMPTY);

        if(empty($array_with_encoded_emails)) {
            // Exit early. We don't have anything to clean.
            return array(
                'strings_deleted' => 0,
                'last_processed_id' => 0,
                'table_end' => true
            );
        }

        $strings_deleted = 0;
        $table_end = false;

        $last_id_query = "SELECT MAX(id) as ID FROM $table_name;";
        $last_table_id = intval($query_component->db->get_var($last_id_query));
        if ($last_processed_id < $last_table_id) {

            // Select the next batch of rows to process
            $query = "SELECT id, original FROM $table_name 
              WHERE id > $last_processed_id 
              ORDER BY id ASC 
              LIMIT $batch_size";
            $batch_rows = $query_component->db->get_results($query, ARRAY_A);

            if ($batch_rows) {
                $ids_to_delete = [];
                foreach ($batch_rows as $row){
                    $original = $row['original'];
                    $decoded_original = html_entity_decode($original);
                    if (in_array($decoded_original, $array_with_encoded_emails)){
                        $ids_to_delete[] = $row['id'];
                    }
                    $last_processed_id =  $row['id'];
                }

                if (!empty($ids_to_delete)) {
                    $ids_string = implode(',', $ids_to_delete);
                    $delete_query = "DELETE FROM $table_name WHERE id IN ($ids_string)";
                    $strings_deleted = $query_component->db->query($delete_query);
                }
            }
        } else {
            $table_end = true;
        }

        return array(
            'strings_deleted' => $strings_deleted,
            'last_processed_id' => $last_processed_id,
            'table_end' => $table_end
        );

    }

    private function table_names(){
        $settings_component = $this->trp->get_component( 'settings' );
        $settings      = $settings_component->get_settings();

        $query_component = $this->trp->get_component( 'query' );

        $default_language = $settings['default-language'];
        $array_of_table_names = $query_component->get_all_table_names( $default_language, array() );
        $originals_table =  $query_component->get_table_name_for_original_strings();

        $array_of_table_names[] = $originals_table;
        return $array_of_table_names;
    }

}