<?php
/**
 * Plugin Name: TranslatePress - Production Automatic Translation Only
 * Plugin URI: https://translatepress.com
 * Description: This plugin ensures that the Automatic Translation feature of TranslatePress is enabled only on the production website (specified in the Production Domain field).
 * Version: 1.0
 * Author: Sandu Denis
 * Author URI: https://translatepress.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

 
// Prevent direct access to the plugin file
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add submenu page under the Tools tab.
 */

 function trpc_add_plugin_action_links($links, $file)
{
    $plugin_file = plugin_basename(__FILE__);

    if ($file == $plugin_file) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('tools.php?page=trpc-production-domain'),
            __('Settings', 'translatepress-multilingual')
        );

        array_unshift($links, $settings_link);
    }

    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'trpc_add_plugin_action_links', 10, 2);

function trpc_add_submenu_page()
{
    add_submenu_page(
        'tools.php',
        __('TranslatePress Production Domain', 'translatepress-multilingual'),
        __('TranslatePress Production Domain', 'translatepress-multilingual'),
        'manage_options',
        'trpc-production-domain',
        'trpc_submenu_page_callback'
    );
}
add_action('admin_menu', 'trpc_add_submenu_page');

/**
 * Render the submenu page content.
 */
function trpc_submenu_page_callback()
{
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        return;
    }

    // Save the production domain
    if (isset($_POST['trpc_production_domain_nonce']) && wp_verify_nonce($_POST['trpc_production_domain_nonce'], 'trpc_production_domain_action')) {
        $production_domain = sanitize_text_field($_POST['trpc_production_domain']);
        $production_domain = preg_replace('/^(https?:\/\/)/', '', $production_domain);
        $production_domain = rtrim($production_domain, '/');
        update_option('trpc_production_domain', $production_domain);
    }


    // Retrieve the production domain from the database
    $production_domain = get_option('trpc_production_domain', '');

    ?>
    <div class="wrap">
        <h1><?php _e('TranslatePress Production Domain', 'translatepress-multilingual'); ?></h1>
        <p><?php _e('Enter the domain of your production website. Automatic translation will be disabled on other domains.', 'translatepress-multilingual'); ?></p>
        <form method="post" action="">
            <?php wp_nonce_field('trpc_production_domain_action', 'trpc_production_domain_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Production Domain', 'translatepress-multilingual'); ?></th>
                    <td>
                        <input type="text" name="trpc_production_domain" value="<?php echo esc_attr($production_domain); ?>" class="regular-text" />
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

/**
 * Disable automatic translation API if the domain is not the specified production domain.
 */
function trpc_production_automatic_translation_only()
{
    $production_domain = get_option('trpc_production_domain', '');

    if ($_SERVER['HTTP_HOST'] != $production_domain) {
        add_filter('trp_disable_automatic_translations_due_to_error', '__return_true');

        $error_message = wp_kses(
            __('Automatic translations are turned off on this website because it is not the main production site. You can change the production domain by <a href="/wp-admin/tools.php?page=trpc-production-domain">clicking here</a>.', 'translatepress-multilingual'),
            array(
                'strong' => array(),
                'a' => array(
                    'href' => array()
                )
            )
        );
        

        // Display the error message in the admin area
        add_action('admin_notices', function () use ($error_message) {
            echo '<div class="notice notice-error"><p>' . $error_message . '</p></div>';
        });
    }
}
add_action('init', 'trpc_production_automatic_translation_only');
